import java.awt.Color;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.FileImageInputStream;
import javax.imageio.stream.MemoryCacheImageOutputStream;


public class CMVEncoder {
	
	
	
	public static void main(String[] args) throws IOException {
		if (args.length != 3) {
			System.out.println("CMVEncoder <jpeg-stream> <wav-file> <cmv-output>");
		}

		CMVEncoder encoder = new CMVEncoder();
		encoder.encode(new File(args[0]), new File(args[1]), new File(args[2]));
		
	}

	private void encode(File jpegs, File wav, File cmv) throws IOException {
		File tmp = File.createTempFile("cmv", ".dat");
		tmp.deleteOnExit();
		Data d = new Data();
		transcodeJPEG(jpegs, tmp, d);
		writeCMV(tmp, wav, cmv, d);
		
	}
	
	
	
	private void writeCMV(File tmp, File wav, File cmv, Data data) throws IOException {
		BufferedOutputStream baos = new BufferedOutputStream(new FileOutputStream(cmv));
		copyWav(baos, wav);
		writeHeader(baos, data);
		writeFrames(baos, tmp, data);
		baos.close();
		System.out.println("Conversion completed!");
		tmp.delete();
	}
	
	
	
	
	private void writeFrames(OutputStream os, File tmp, Data d) throws IOException {
		System.out.println("Copying video...");
		DataInputStream dis = new DataInputStream(new BufferedInputStream(new FileInputStream(tmp)));
		
		byte[] buffer = new byte[d.chunkSize];
		
		int size = dis.readInt();
		while (size > 0) {
			dis.readFully(buffer, 0, size);
			wT(os, size);
			os.write(buffer);
			Arrays.fill(buffer, (byte)0);
			
			size = dis.readInt();
		}
		
		dis.close();
	}

	private static void wT(OutputStream os, int value) throws IOException {
		os.write(value & 0xff);
		os.write((value >> 8) & 0xff);
		os.write((value >> 16) & 0xff);
	}
	
	private void writeHeader(OutputStream os, Data data) throws IOException {
		System.out.println("Writing header...");
		os.write("CMV001000".getBytes());
		wT(os, 320);	// width
		wT(os, 240);	// height
		wT(os, 16);		// w-step-size
		wT(os, 16);		// h-step-size
		wT(os, data.chunkSize + 3);
		wT(os, 25);		// fps / frames per chunk
		wT(os, data.frames);
		wT(os, 1);
		wT(os, 0);
		wT(os, 25);		// fps / frames per chunk
		wT(os, 0x010101);	// reserved
		wT(os, 0x010101);	// reserved
		wT(os, 0x010101);	// reserved
		wT(os, 0x010101);	// reserved
	}
	

	private void copyWav(OutputStream os, File wav) throws IOException {
		System.out.println("Copying audio...");
		BufferedInputStream bis = new BufferedInputStream(new FileInputStream(wav));
		byte[] buffer = new byte[1024 * 256];	// declare 256K buffer
		int r = bis.read(buffer);
		while (r >= 0) {
			if (r > 0) {
				os.write(buffer, 0, r);
			} else {
				Thread.yield();
			}
			r = bis.read(buffer);
		}
		bis.close();
	}

	private void transcodeJPEG(File jpegsIn, File jpegsOut, Data data) throws IOException {
		System.out.println("Transcoding video...");
		JPEGReader jr = new JPEGReader(jpegsIn);
		ImageWriter iw = ImageIO.getImageWritersByFormatName("jpeg").next();
		ImageWriteParam iwp = iw.getDefaultWriteParam();
		iwp.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
		iwp.setCompressionQuality(0.3f);
		
		int framesPerChunk = 25;
		int width = jr.width();
		int height = jr.height();
		DataOutputStream dos = new DataOutputStream(new FileOutputStream(jpegsOut));
		int s_height = (height / 16) * 16;
		int c_height = s_height * framesPerChunk;
		int cc = 0;
		int maxSize = 0;
		BufferedImage out = new BufferedImage(width, c_height, BufferedImage.TYPE_INT_ARGB);
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		MemoryCacheImageOutputStream mcios = new MemoryCacheImageOutputStream(baos);
		iw.setOutput(mcios);
		
		Graphics g = out.getGraphics();
		g.setColor(Color.BLACK);
		g.fillRect(0, 0, width, c_height);
		int chunks = 0;
		int frames = 0;
		while (jr.hasNext()) {
			BufferedImage frame = jr.next();
			g.drawImage(frame, 0, s_height * cc, null);
			cc++;
			frames ++;
			if (cc >= framesPerChunk) {
				g.dispose();
				System.out.print(".");
				chunks++;
				if (chunks % 60 == 0) {
					System.out.println();	
				}
				IIOImage image = new IIOImage(out, null, null);
				iw.write(null, image, iwp);
				mcios.flush();
				if (baos.size() > maxSize) {
					maxSize = baos.size();
				}
				dos.writeInt(baos.size());
				baos.writeTo(dos);
				baos.reset();
				cc = 0;
				g = out.getGraphics();
			}
		}
		g.dispose();
		if (cc > 0) {
			IIOImage image = new IIOImage(out, null, null);
			iw.write(null, image, iwp);
			mcios.flush();
			if (baos.size() > maxSize) {
				maxSize = baos.size();
			}
			dos.writeInt(baos.size());
			baos.writeTo(dos);
		}
		mcios.close();
		baos.close();
		jr.close();
		dos.writeInt(0);
		iw.dispose();
		dos.close();
		System.out.println("Done!");		
				
		data.chunkSize = maxSize;
		data.chunks = chunks;
		data.frames = frames;
	}


	private class Data {
		public int chunkSize;
		public int chunks;
		public int frames;
	}

	private class JPEGReader {
		
		private BufferedImage _current;
		private ImageReader _reader;
		private FileImageInputStream _is;
		private int _index;
		private int _max;
		
		public JPEGReader(File jpegsIn) throws IOException {
			_is = new FileImageInputStream(jpegsIn);
			_reader = ImageIO.getImageReadersByFormatName("jpeg").next();
			_reader.setInput(_is);
			_max = _reader.getNumImages(true);
			
			readNext();
		}
		
		public int width() {
			return _current.getWidth();			
		}
		
		public int height() {
			return _current.getHeight();
		}
		
		private void readNext() throws IOException {
			if (_index >= _max) {
				_current = null;
				return;
			}
			_current = _reader.read(_index++);
			
		}

		public boolean hasNext() {
			return _current != null;
		}
		
		public BufferedImage next() throws IOException {
			BufferedImage tmp = _current;
			readNext();
			return tmp;
		}
		
		public void close() throws IOException {
			_is.close();
		}
		
		
		
	}

}
