import uuid

from yt_dlp.extractor.common import InfoExtractor
from yt_dlp.utils import ExtractorError, traverse_obj

class PlutoTV2IE(InfoExtractor):
    _VALID_URL = (
        r'https://(?:www\.)?pluto\.tv(?:/[a-z]{2,3})?'
        r'(?:/on-demand)?'
        r'/(?P<type>movies|series)'
        r'/(?P<id>[a-z0-9-]+)'
        r'(?:(?:/season/(\d+)/episode/(?P<episode>[a-z0-9-]+)))?'
    )

    _AUTH_URL = 'https://boot.pluto.tv/v4/start'
    _SERIES_URL = 'https://service-vod.clusters.pluto.tv/v4/vod/series/{series_id}/seasons'
    _MOVIE_URL = 'https://service-vod.clusters.pluto.tv/v4/vod/items?ids={movie_id}'
    _EPISODES_URL = 'http://api.pluto.tv/v2/episodes/{video_id}/clips.json'
    _LICENSE_URL = 'https://service-concierge.clusters.pluto.tv/v1/wv/alt?jwt='

    def _real_initialize(self):
        self.token = None
        self.auth_headers = None

    def _authenticate(self):
        # JWT is valid for 24h, should be enough for most jobs
        if self.token and self.auth_headers:
            return

        params = {
            'appName': 'web',
            'appVersion': 'na',
            'clientID': str(uuid.uuid1()),
            'deviceDNT': 0,
            'deviceId': 'unknown',
            'clientModelNumber': 'na',
            'serverSideAds': 'false',
            'deviceMake': "unknown",
            'deviceModel': 'web',
            'deviceType': 'web',
            'deviceVersion': 'unknown',
            'sid': str(uuid.uuid1()),
        }

        info = self._download_json(self._AUTH_URL, None, query=params, note='Authenticate')
        self.token = info.get('sessionToken')
        self.auth_headers = { 'Authorization': f'Bearer {self.token}' }

    def _real_extract(self, url):
        type, id, episode_id = self._match_valid_url(url).group('type', 'id', 'episode')
        self._authenticate()

        if type == 'series' and episode_id:
            return self._extract_series(id, episode_id)

        elif type == 'movies':
            return self._extract_movies(id)

        return None

    def _extract_series(self, series_id, episode_id):
        details = self._download_json(
            self._SERIES_URL.format(series_id=series_id), None,
            note='Episode Metadata', headers=self.auth_headers)
        seasons = traverse_obj(details, ('seasons', ..., 'episodes'))
        ep = next((ep for season in seasons for ep in season if ep.get('_id') == episode_id), None)
        thumbnail = next((x.get('url') for x in ep.get('covers') if x.get('aspectRatio') == '16:9'), None)

        return {
            'id': episode_id,
            'formats': self.get_tracks(episode_id),
            'series': details.get('name'),
            'thumbnail': thumbnail,
            **traverse_obj(ep, {
                'title': 'name',
                'episode': 'name',
                'description': 'description',
                'season_number': 'season',
                'episode_number': 'number',
            })
        }

    def _extract_movies(self, movie_id):
        detail = self._download_json(
            self._MOVIE_URL.format(movie_id=movie_id), None,
            note='Details', headers=self.auth_headers)
        movie = traverse_obj(detail, 0)
        thumbnail = next((x.get('url') for x in movie.get('covers') if x.get('aspectRatio') == '16:9'), None)

        return {
            'id': movie_id,
            'formats': self.get_tracks(movie_id),
            'thumbnail': thumbnail,
            'title': movie.get('name'),
            'description': movie.get('description'),
        }

    def get_tracks(self, video_id):
        url = self._EPISODES_URL.format(video_id=video_id)
        video = self._download_json(url, None, headers=self.auth_headers)

        sources = next((item.get('sources') for item in video if not self.bumpers(item.get('name', ''))), None)

        if not sources:
            raise ValueError('Unable to find manifest for this title')

        hls = next((x.get('file') for x in sources if x.get('type').lower() == 'hls'), None)
        dash = next((x.get('file') for x in sources if x.get('type').lower() == 'dash'), None)

        if hls:
            m3u8_url = hls.replace('https://siloh.pluto.tv', 'http://silo-hybrik.pluto.tv.s3.amazonaws.com')
            res = self._download_webpage_handle(
                m3u8_url, video_id,
                note='Downloading m3u8 information',
                errnote='Failed to download m3u8 information')
            if res is False:
                return [], {}

            m3u8_doc, urlh = res
            m3u8_url = urlh.url

            tracks, subtitles = self._parse_m3u8_formats_and_subtitles(
                self.clean_manifest(m3u8_doc), m3u8_url, video_id=video_id)

        else:
            manifest_url = dash.replace('https://siloh.pluto.tv', 'http://silo-hybrik.pluto.tv.s3.amazonaws.com')
            tracks, subtitles = self._extract_mpd_formats_and_subtitles(manifest_url, video_id)

        return tracks

    @staticmethod
    def clean_manifest(text: str) -> str:
        # Remove fairplay entries
        index = text.find('#PLUTO-DRM:ID="fairplay')
        if index == -1:
            return text
        else:
            end_of_previous_line = text.rfind('\n', 0, index)
            if end_of_previous_line == -1:
                return ""
            else:
                return text[:end_of_previous_line]

    @staticmethod
    def bumpers(text: str) -> bool:
        ads = (
            'Pluto_TV_OandO',
            '_ad',
            'creative',
            'Bumper',
            'Promo',
            'WarningCard',
        )
        return any(ad in text for ad in ads)

